-- time_based_logger.lua
obs = obslua

-- データ保持用
markers = {}

-- 設定値
save_path = ""
start_time_str = "00:00" -- ユーザーが入力する開始時間 (HH:MM)
notify_source_name = "" -- 通知用ソース（空欄なら無効）

-- 通知消去用タイマー
notify_timer = 0
NOTIFY_DURATION = 2.0

function script_description()
	return [[
【時間指定型】YouTubeマーカー記録スクリプト

設定した「開始時間（例 22:00）」を 00:00:00 とみなして、
キーを押した時点（-5秒）の経過時間を記録します。

・ボタン押し忘れの心配がありません。
・配信終了時にテキストファイルとして書き出します。
]]
end

function script_properties()
	local props = obs.obs_properties_create()
	
	-- 設定項目
	obs.obs_properties_add_path(props, "save_path", "保存先フォルダ", obs.OBS_PATH_DIRECTORY, "", nil)
	
	local p_time = obs.obs_properties_add_text(props, "start_time_str", "配信開始時間 (HH:MM)", obs.OBS_TEXT_DEFAULT)
	obs.obs_property_set_long_description(p_time, "YouTubeの配信開始予定時刻を 24時間表記で入力してください。\n例: 22:00, 09:30")

	local p_notify = obs.obs_properties_add_text(props, "notify_source_name", "通知用テキストソース名 (任意)", obs.OBS_TEXT_DEFAULT)
	obs.obs_property_set_long_description(p_notify, "画面に「Saved!」と出したい場合のみ指定。\nソースを「非表示」にすれば配信にも自分にも見えませんが、エラーにはなりません。")
	
	return props
end

function script_update(settings)
	save_path = obs.obs_data_get_string(settings, "save_path")
	start_time_str = obs.obs_data_get_string(settings, "start_time_str")
	notify_source_name = obs.obs_data_get_string(settings, "notify_source_name")
end

-- 秒数を HH:MM:SS に変換
function format_seconds(total_seconds)
	if total_seconds < 0 then total_seconds = 0 end
	local h = math.floor(total_seconds / 3600)
	local m = math.floor((total_seconds % 3600) / 60)
	local s = math.floor(total_seconds % 60)
	return string.format("%02d:%02d:%02d", h, m, s)
end

-- テキストソース更新
function set_text_source_content(source_name, text)
	if source_name == "" then return end
	local source = obs.obs_get_source_by_name(source_name)
	if source ~= nil then
		local settings = obs.obs_data_create()
		obs.obs_data_set_string(settings, "text", text)
		obs.obs_source_update(source, settings)
		obs.obs_data_release(settings)
		obs.obs_source_release(source)
	end
end

-- 毎フレーム処理（通知消去用）
function script_tick(seconds)
	if notify_timer > 0 then
		notify_timer = notify_timer - seconds
		if notify_timer <= 0 then
			set_text_source_content(notify_source_name, "")
			notify_timer = 0
		end
	end
end

-- マーカー追加のメイン処理
function add_marker(pressed)
	if not pressed then return end
	
	-- 現在のシステム時間を取得
	local now = os.time()
	local date_table = os.date("*t", now) -- 現在の年月日時分秒

	-- 設定された開始時間 (HH:MM) を解析
	local start_h, start_m = start_time_str:match("(%d+):(%d+)")
	
	if start_h == nil or start_m == nil then
		print("エラー: 時間設定が正しくありません (例: 22:00)")
		return
	end

	-- 今日の日付で「開始時間」のタイムスタンプを作成
	local target_time = os.time({
		year = date_table.year,
		month = date_table.month,
		day = date_table.day,
		hour = start_h,
		min = start_m,
		sec = 0
	})

	-- 【日またぎ対応】
	-- もし「現在時刻」より「設定した開始時刻」の方が未来にある場合
	-- (例: 今が深夜 01:00 で、設定が 23:00 の場合、設定時刻は昨日の23:00とみなす)
	if target_time > now then
		target_time = target_time - (24 * 3600) -- 1日(86400秒)引く
	end

	-- 経過秒数を計算
	local elapsed = os.difftime(now, target_time)
	
	-- 5秒前 (-5秒)
	local mark_seconds = elapsed - 5
	
	-- 記録作成
	local time_str = format_seconds(mark_seconds)
	table.insert(markers, time_str)
	
	local log_msg = string.format("Marker: %s (設定開始時間 %s:%s からの経過)", time_str, start_h, start_m)
	print(log_msg)

	-- 通知
	set_text_source_content(notify_source_name, "Saved! [" .. time_str .. "]")
	notify_timer = NOTIFY_DURATION
end

-- ファイル保存処理
function save_to_file()
	if #markers == 0 then return end
	if save_path == "" then return end

	local date_str = os.date("%Y-%m-%d_%H-%M-%S")
	local filepath = save_path .. "/Youtube_TimeMarkers_" .. date_str .. ".txt"

	local file = io.open(filepath, "w")
	if file then
		file:write("--- YouTube 配信マーカー (時間指定: " .. start_time_str .. " スタート) ---\n")
		file:write("※各タイムスタンプはボタン押下の5秒前です\n\n")
		for i, timestamp in ipairs(markers) do
			file:write(string.format("%s - ポイント%d\n", timestamp, i))
		end
		file:close()
		print("保存完了: " .. filepath)
		markers = {} -- リセット
	end
end

-- 配信終了検知
function on_event(event)
	if event == obs.OBS_FRONTEND_EVENT_STREAMING_STOPPED then
		save_to_file()
	elseif event == obs.OBS_FRONTEND_EVENT_STREAMING_STARTED then
		markers = {} -- 念のためリセット
	end
end

function script_load(settings)
	hk = obs.obs_hotkey_register_frontend("save_marker_time_based", "[Marker] おもしろポイント記録 (時間指定)", add_marker)
	local arr = obs.obs_data_get_array(settings, "save_marker_time_based")
	obs.obs_hotkey_load(hk, arr)
	obs.obs_data_array_release(arr)

	obs.obs_frontend_add_event_callback(on_event)
end

function script_save(settings)
	local arr = obs.obs_hotkey_save(hk)
	obs.obs_data_set_array(settings, "save_marker_time_based", arr)
	obs.obs_data_array_release(arr)
end